<?php
// app/Http/Controllers/DesaController.php

namespace App\Http\Controllers;

use App\Models\Desa;
use App\Models\Dusun;
use Illuminate\Http\Request;

class DesaController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth', 'admin']);
    }
    
    /**
     * Display listing of desa
     */
    public function index()
    {
        $desa = Desa::with('dusun')->paginate(15);
        return view('admin.desa', compact('desa'));
    }
    
    /**
     * Store new desa
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nama_desa' => 'required|string|max:255',
            'kecamatan' => 'required|string|max:255',
            'kabupaten' => 'nullable|string|max:255',
            'alamat' => 'nullable|string',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'jumlah_penduduk' => 'nullable|integer|min:0',
            'kebutuhan_air_harian' => 'nullable|integer|min:0',
        ]);
        
        $validated['kabupaten'] = $validated['kabupaten'] ?? 'Gunungkidul';
        
        Desa::create($validated);
        
        return redirect()->back()->with('success', 'Desa berhasil ditambahkan!');
    }
    
    /**
     * Update desa
     */
    public function update(Request $request, $id)
    {
        $validated = $request->validate([
            'nama_desa' => 'required|string|max:255',
            'kecamatan' => 'required|string|max:255',
            'kabupaten' => 'nullable|string|max:255',
            'alamat' => 'nullable|string',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
            'jumlah_penduduk' => 'nullable|integer|min:0',
            'kebutuhan_air_harian' => 'nullable|integer|min:0',
        ]);
        
        $desa = Desa::findOrFail($id);
        $desa->update($validated);
        
        return redirect()->back()->with('success', 'Desa berhasil diperbarui!');
    }
    
    /**
     * Delete desa
     */
    public function destroy($id)
    {
        $desa = Desa::findOrFail($id);
        $desa->delete();
        
        return redirect()->back()->with('success', 'Desa berhasil dihapus!');
    }
    
    /**
     * Get desa details with JSON
     */
    public function show($id)
    {
        $desa = Desa::with(['dusun', 'prediksi' => function($query) {
            $query->where('tanggal_prediksi', '>=', now()->format('Y-m-d'))
                  ->orderBy('tanggal_prediksi', 'asc')
                  ->limit(7);
        }])->findOrFail($id);
        
        return response()->json($desa);
    }
    
    /**
     * Store dusun
     */
    public function storeDusun(Request $request)
    {
        $validated = $request->validate([
            'desa_id' => 'required|exists:desa,id',
            'nama_dusun' => 'required|string|max:255',
            'jumlah_rt' => 'nullable|integer|min:0',
            'jumlah_rw' => 'nullable|integer|min:0',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
        ]);
        
        Dusun::create($validated);
        
        return redirect()->back()->with('success', 'Dusun berhasil ditambahkan!');
    }
    
    /**
     * Update dusun
     */
    public function updateDusun(Request $request, $id)
    {
        $validated = $request->validate([
            'nama_dusun' => 'required|string|max:255',
            'jumlah_rt' => 'nullable|integer|min:0',
            'jumlah_rw' => 'nullable|integer|min:0',
            'latitude' => 'nullable|numeric|between:-90,90',
            'longitude' => 'nullable|numeric|between:-180,180',
        ]);
        
        $dusun = Dusun::findOrFail($id);
        $dusun->update($validated);
        
        return redirect()->back()->with('success', 'Dusun berhasil diperbarui!');
    }
    
    /**
     * Delete dusun
     */
    public function destroyDusun($id)
    {
        $dusun = Dusun::findOrFail($id);
        $dusun->delete();
        
        return redirect()->back()->with('success', 'Dusun berhasil dihapus!');
    }
}
